package furny.swing.admin.genotype;

import furny.ga.FurnLayoutIndividual;
import furny.ga.FurnLayoutPhenotypeGenerator;
import furny.ga.rules.FurnitureIntersectionRule;
import furny.ga.rules.RoomContainsRule;
import furny.ga.util.FurnLayoutIOUtil;
import furny.jme.FurnyApplication;
import ga.core.GA;
import ga.core.validation.GAContext;
import ga.core.validation.RuleValidator;
import ga.view.streaming.ShowRoomState;
import ga.view.streaming.showroom.BoxShowRoom;
import ga.view.streaming.showroom.CameraSettings;
import ga.view.streaming.showroom.ShowRoom;
import ga.view.streaming.showroom.ShowRoomSettings;
import ga.view.streaming.showroom.ShowRoomSettings.ShowRoomType;

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.io.File;
import java.util.Locale;
import java.util.concurrent.Callable;

import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTextArea;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.Timer;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import com.jme3.app.SimpleApplication;
import com.jme3.system.AppSettings;
import com.jme3.system.JmeCanvasContext;

/**
 * This is a panel that allows to display, create, manipulate, load and save a
 * genotype and corresponding phenotype.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class GenotypePanel extends JPanel implements ChangeListener {
  private Canvas canvas;

  private SimpleApplication app;
  private final AppSettings settings;

  private ShowRoomState<FurnLayoutIndividual> srState;

  private final FurnLayoutPhenotypeGenerator phenotypeGenerator = new FurnLayoutPhenotypeGenerator();
  private RuleValidator<FurnLayoutIndividual> validator;

  private ShowRoom showRoom;

  private boolean initialized;
  private boolean showRoomNeedsUpdate;

  private JTextArea genotypeField;
  private JSpinner widthSpinner, lengthSpinner;
  private JSpinner minCountSpinner, maxCountSpinner;

  private int roomWidth = 500;
  private int roomLength = 500;

  private int minCount = 2;
  private int maxCount = 5;

  private final GAContext context = new GAContext();
  private final FurnLayoutIndividual ind = new FurnLayoutIndividual(context);

  private Timer timer;

  private final File lastDir = new File(".");

  /**
   * Instantiates a new genotype panel.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public GenotypePanel() {
    settings = new AppSettings(true);
    settings.setRenderer(AppSettings.LWJGL_OPENGL2);
    settings.setTitle("");

    Locale.setDefault(Locale.ENGLISH);

    checkInit();
  }

  @Override
  public void stateChanged(final ChangeEvent e) {
    // from spinner models changeListener

    final int newWidth = (Integer) widthSpinner.getValue();
    final int newLength = (Integer) lengthSpinner.getValue();

    if (newWidth != roomWidth || newLength != roomLength) {
      roomWidth = newWidth;
      roomLength = newLength;

      showRoomNeedsUpdate = true;
      timer.restart();
    }
  }

  /**
   * Checks if required components have been initialized.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void checkInit() {
    if (!initialized) {

      app = new FurnyApplication();

      app.setSettings(settings);

      app.enqueue(new Callable<Void>() {
        @Override
        public Void call() throws Exception {
          validator = new RuleValidator<FurnLayoutIndividual>();
          validator.addRule(new FurnitureIntersectionRule());
          validator.addRule(new RoomContainsRule());

          final ShowRoomSettings srSettings = new ShowRoomSettings();
          srSettings.put(ShowRoomSettings.TYPE, ShowRoomType.BOX);
          srSettings.put(ShowRoomSettings.BOX_WIDTH, roomWidth / 100f);
          srSettings.put(ShowRoomSettings.BOX_LENGTH, roomLength / 100f);
          srSettings.put(ShowRoomSettings.BOX_HEIGHT, 2.6f);

          showRoom = new BoxShowRoom(app.getAssetManager(), settings,
              srSettings);
          context.put(GA.KEY_VALIDATION_SPACE, showRoom);

          context.put(GA.KEY_GENOME_MIN_LENGTH, minCount);
          context.put(GA.KEY_GENOME_MAX_LENGTH, maxCount);

          phenotypeGenerator.setAssetManager(app.getAssetManager());

          final CameraSettings camSettings = CameraSettings
              .getOrthographicSettings(showRoom, settings);

          srState = new ShowRoomState<FurnLayoutIndividual>(null, camSettings);
          srState.setSettings(settings);

          app.getStateManager().attach(srState);

          return null;
        }
      });

      app.setPauseOnLostFocus(false);
      app.setShowSettings(false);

      app.createCanvas();

      final JmeCanvasContext context = (JmeCanvasContext) app.getContext();
      canvas = context.getCanvas();

      // canvas.addComponentListener(new ComponentAdapter() {
      // @Override
      // public void componentResized(final ComponentEvent e) {
      // updateViewerSize();
      // }
      //
      // @Override
      // public void componentShown(final ComponentEvent e) {
      // updateViewerSize();
      // }
      // });

      app.startCanvas();

      app.enqueue(new Callable<Void>() {
        @Override
        public Void call() throws Exception {

          new SwingWorker<Void, Void>() {
            @Override
            protected Void doInBackground() throws Exception {
              Thread.sleep(1000);
              return null;
            }

            @Override
            protected void done() {
              invalidate();
              validate();
              randomizeInd();
            }
          }.execute();

          srState.setRootNode(showRoom);

          return null;
        }
      });

      setLayout(new GridBagLayout());
      final GridBagConstraints constraints = new GridBagConstraints();
      constraints.insets = new Insets(5, 5, 5, 5);
      constraints.gridx = 0;
      constraints.gridy = 0;
      constraints.weightx = 1.0d;
      constraints.weighty = 1.0d;
      constraints.fill = GridBagConstraints.BOTH;
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.gridheight = 1;

      add(createGenotypePanel(), constraints);

      constraints.gridwidth = 1;
      constraints.gridheight = 1;
      constraints.gridx = 0;
      constraints.gridy++;
      constraints.weightx = 0.0d;
      constraints.weighty = 0.0d;

      add(new JLabel("Room Width (cm)"), constraints);

      constraints.gridx++;

      SpinnerNumberModel model = new SpinnerNumberModel(roomWidth, 50, 5000, 5);

      widthSpinner = new JSpinner(model);
      model.addChangeListener(this);

      add(widthSpinner, constraints);

      constraints.gridx++;

      add(new JLabel("Room Length (cm)"), constraints);

      constraints.gridx++;

      model = new SpinnerNumberModel(roomLength, 50, 5000, 5);
      lengthSpinner = new JSpinner(model);

      model.addChangeListener(this);

      add(lengthSpinner, constraints);

      constraints.gridx++;
      add(Box.createHorizontalGlue(), constraints);

      constraints.gridx = 0;
      constraints.gridy++;
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.weightx = 1.0d;
      constraints.weighty = 1.0d;

      canvas.setPreferredSize(new Dimension(800, 400));
      canvas.setMaximumSize(canvas.getPreferredSize());
      canvas.setSize(canvas.getPreferredSize());

      add(canvas, constraints);

      updateGenotypeField();

      addFocusListener(new FocusAdapter() {
        @Override
        public void focusGained(final FocusEvent e) {
          checkInit();
        }
      });

      invalidate();
      validate();
      repaint();

      timer = new Timer(2000, new ActionListener() {
        @Override
        public void actionPerformed(final ActionEvent e) {
          updateShowroom();
        }
      });
      timer.setRepeats(true);
      timer.start();

      initialized = true;
    }
  }

  /**
   * Creates the min max count panel.
   * 
   * @return the new panel
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private JPanel createCountPanel() {
    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());

    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.fill = GridBagConstraints.BOTH;

    panel.add(new JLabel("Min Count"), constraints);

    constraints.gridx++;

    SpinnerNumberModel model = new SpinnerNumberModel(minCount, 1, 12, 1);

    minCountSpinner = new JSpinner(model);

    model.addChangeListener(this);

    panel.add(minCountSpinner, constraints);

    constraints.gridx = 0;
    constraints.gridy++;

    panel.add(new JLabel("Max Count"), constraints);

    constraints.gridx++;

    model = new SpinnerNumberModel(maxCount, 1, 12, 1);
    maxCountSpinner = new JSpinner(model);

    model.addChangeListener(this);

    panel.add(maxCountSpinner, constraints);

    return panel;
  }

  /**
   * Creates the genotype panel.
   * 
   * @return the new panel
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private JPanel createGenotypePanel() {
    final JPanel panel = new JPanel();

    panel.setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.weightx = 1.0d;
    constraints.weighty = 0.0d;
    constraints.gridwidth = 1;
    constraints.gridheight = 3;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.anchor = GridBagConstraints.NORTHEAST;

    genotypeField = new JTextArea(5, 80);
    genotypeField.setLineWrap(true);
    genotypeField.setWrapStyleWord(true);
    genotypeField.setMinimumSize(new Dimension(800, 100));

    panel.add(genotypeField, constraints);

    constraints.weightx = 0.0d;
    constraints.weighty = 0.0d;
    constraints.gridheight = 1;
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.anchor = GridBagConstraints.NORTHEAST;
    constraints.gridx++;
    panel.add(createCountPanel(), constraints);

    constraints.gridy++;
    panel.add(new JButton(new ActionCreateRandomGenotype()), constraints);

    constraints.gridx++;
    panel.add(new JButton(new ActionLoadGenotype()), constraints);

    constraints.gridx = 1;
    constraints.gridy++;
    panel.add(new JButton(new ActionUpdateView()), constraints);

    constraints.gridx++;
    panel.add(new JButton(new ActionSaveGenotype()), constraints);

    return panel;
  }

  /**
   * Updates the genotype text field.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void updateGenotypeField() {
    genotypeField.setText(ind.getGenotypeString());
  }

  /**
   * Updates the showroom size.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void updateShowroom() {
    if (showRoomNeedsUpdate && initialized) {
      showRoomNeedsUpdate = false;
      app.enqueue(new Callable<Void>() {
        @Override
        public Void call() throws Exception {

          final ShowRoomSettings srSettings = new ShowRoomSettings();
          srSettings.put(ShowRoomSettings.TYPE, ShowRoomType.BOX);
          srSettings.put(ShowRoomSettings.BOX_WIDTH, roomWidth / 100f);
          srSettings.put(ShowRoomSettings.BOX_LENGTH, roomLength / 100f);
          srSettings.put(ShowRoomSettings.BOX_HEIGHT, 2.6f);

          showRoom = new BoxShowRoom(app.getAssetManager(), settings,
              srSettings);
          context.put(GA.KEY_VALIDATION_SPACE, showRoom);

          final CameraSettings camSettings = CameraSettings
              .getOrthographicSettings(showRoom, settings);

          srState.setEnabled(false);
          srState.setCamSettings(camSettings);
          srState.setRootNode(showRoom);

          showRoom.setPhenotype(phenotypeGenerator.createPhenotype(ind));

          new SwingWorker<Void, Void>() {

            @Override
            protected Void doInBackground() throws Exception {
              Thread.sleep(1000);
              return null;
            }

            @Override
            protected void done() {
              if (!validator.isValid(ind, context)) {
                randomizeInd();
              }

              invalidate();
              validate();
              revalidate();
            }
          }.execute();

          return null;
        }
      });
    }
  }

  /**
   * Updates the displayed phenotype.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void updateView() {
    app.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        showRoom.setPhenotype(phenotypeGenerator.createPhenotype(ind));
        srState.setRootNode(showRoom);
        // srState.setCamSettings(srState.g);
        return null;
      }
    });
  }

  /**
   * Randomizes the individual.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void randomizeInd() {
    if (validator != null) {
      int i = 100000;

      do {
        ind.initRandomly();
        if (i-- < 0) {
          ind.getFurnitures().clear();
          break;
        }

      } while (!validator.isValid(ind, context));
      updateGenotypeField();
      updateView();
    }
  }

  /**
   * Action to create a random genotype.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionCreateRandomGenotype extends AbstractAction {

    /**
     * Instantiates a new action for creating random genotypes.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionCreateRandomGenotype() {
      super("Random genotype");
      putValue(SHORT_DESCRIPTION, "Create a random genotype");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      minCount = (Integer) minCountSpinner.getValue();
      maxCount = (Integer) maxCountSpinner.getValue();

      minCount = Math.min(minCount, maxCount);

      minCountSpinner.setValue(minCount);

      context.put(GA.KEY_GENOME_MIN_LENGTH, minCount);
      context.put(GA.KEY_GENOME_MAX_LENGTH, maxCount);

      randomizeInd();
    }
  }

  /**
   * Action to save a genotype to a file.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionSaveGenotype extends AbstractAction {

    /**
     * Instantiates a new action to save genotypes.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionSaveGenotype() {
      super("Save");
      putValue(SHORT_DESCRIPTION, "Save this genotype to a file");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      if (validator.isValid(ind, context)) {
        FurnLayoutIOUtil.saveGenotype(GenotypePanel.this, ind);
      } else {
        JOptionPane.showMessageDialog(GenotypePanel.this,
            "Individual is invalid, create a valid individual first!",
            "Error writing file", JOptionPane.WARNING_MESSAGE);
      }
    }
  }

  /**
   * Action to load genotypes from a file.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionLoadGenotype extends AbstractAction {

    /**
     * Instantiates a new action to load genotypes.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionLoadGenotype() {
      super("Load");
      putValue(SHORT_DESCRIPTION, "Load a genotype from a file");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      final FurnLayoutIndividual newInd = FurnLayoutIOUtil.loadGenotype(
          GenotypePanel.this, ind);
      if (newInd != null) {
        updateGenotypeField();
        updateView();
      }
    }
  }

  /**
   * Action to update the view/phenotype.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionUpdateView extends AbstractAction {

    /**
     * Instantiates a new action update view.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionUpdateView() {
      super("Update View");
      putValue(SHORT_DESCRIPTION, "Create a random genotype");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      if (FurnLayoutIOUtil.parse(ind, genotypeField.getText())
          && validator.isValid(ind, context)) {
        updateGenotypeField();
        updateView();
      } else {
        // flash 5 times on error
        final SwingWorker<Void, Void> sw = new SwingWorker<Void, Void>() {

          @Override
          protected Void doInBackground() throws Exception {
            for (int i = 0; i < 5; i++) {
              SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {
                  genotypeField.setForeground(Color.RED);
                }
              });

              Thread.sleep(300);

              SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {
                  genotypeField.setForeground(Color.RED.darker().darker());
                }
              });

              Thread.sleep(200);
            }
            return null;
          }

          @Override
          protected void done() {
            genotypeField.setForeground(Color.BLACK);
          }
        };
        sw.execute();
      }
    }
  }
}
